<?php
/*
 * Create an animated GIF from multiple images
 *
 * @version 1.0
 * @link https://github.com/Sybio/GifCreator
 * @author Sybio (Clément Guillemain  / @Sybio01)
 * @license http://opensource.org/licenses/gpl-license.php GNU Public License
 * @copyright Clément Guillemain
 * @some small modifications by nageniil (wibcms.de)
 */
class GifCreator {
    private $gif;			// string:	created gif binary
	private $version;		// string:	Encoder version
    private $imgBuilt;		// bool:	Check the image is built or not
	private $frameSources;	// array(single gif binaries)
	private $loop;			// integer: number of loops, 0 means endless
	private $dis;			// integer: dis
	private $colour;		// integer: transparent color
	private $errors;		// array(string): predefined error messages
	public  $ErrNo;			// integer: index of error

    // Methods
    // ===================================================================================
    /**
     * Constructor
     */
    public function __construct() {
        $this->reset();
        $this->version = 'GifCreator 1.0';
        $this->errors = [
            1 => 'Does not support function for only one image.',
    		2 => 'Source is not a GIF image.',
    		3 => 'You have to give resource image variables, image URL or image binary sources in $frames array.',
    		4 => 'Can not create animation from animated GIF source.'
        ];
    }

	/**
     * Create the GIF string
     * 
     * @param array $frames: An array of frame: can be file paths, resource image variables, binary sources or image URLs
     * @param array $durations: An array containing the duration of each frame in 1/100 seconds
     * @param integer $loop: Number of GIF loops before stopping animation (Set 0 to get an infinite loop)
     * 
     * @return string: The GIF binary source or bool: FALSE if no GIF is created
     */
	public function create($frames = array(), $durations = array(), $loop = 0)
    {
		if (!is_array($frames) || !is_array($durations)) {
			$this->ErrNo = 1;
            return false;
		}

		$this->loop = ($loop > -1) ? $loop : 0;
		$this->dis = 2;

//		for ($i = 0; $i < count($frames); $i++) {
		foreach($frames as $i=>$frame) {

			if (is_resource($frame)) { // Resource var

                $resourceImg = $frame;

            } elseif (is_string($frame)) { // File path or URL or Binary source

                if (file_exists($frame) || filter_var($frame, FILTER_VALIDATE_URL)) { // File path
                    $frame = file_get_contents($frame);                    
                }
                $resourceImg = imagecreatefromstring($frame);

			} else { // Fail
				$this->ErrNo = 3;
                return false;
			}

		// assemble frameSources
			ob_start();
			imagegif($resourceImg);
			$frameSource = ob_get_contents();
			$this->frameSources[] = $frameSource;
			ob_end_clean();

            if ($i == 0) {	// the first image
                $colour = imagecolortransparent($resourceImg);
            }

			if (substr($frameSource, 0, 6) != 'GIF87a' && substr($frameSource, 0, 6) != 'GIF89a') {
				$this->ErrNo = 2;
				return false;
			}

			for ($j = (13 + 3 * (2 << (ord($frameSource[10]) & 0x07))), $k = TRUE; $k; $j++) {

				switch ($frameSource[$j]) {

					case '!':
						if ((substr($frameSource, ($j + 3), 8)) == 'NETSCAPE') {
							$this->ErrNo = 4;
							return false;
						}
					break;

					case ';':
						$k = false;
					break;
				}
			}

            unset($resourceImg);
		}	// gathering frames from array completed

        if (isset($colour)) {
            $this->colour = $colour;
        } else {
            $red = $green = $blue = 0;
            $this->colour = ($red > -1 && $green > -1 && $blue > -1) ? ($red | ($green << 8) | ($blue << 16)) : -1;
        }

	// Complete the creation:
		$this->gifAddHeader();

		for ($i = 0; $i < count($this->frameSources); $i++) {
			$this->addGifFrames($i, $durations[$i]);
		}

		$this->gifAddFooter();
        return $this->gif;
	}

    // Internals
    // ===================================================================================
	/**
     * Add the header gif string in its source
     */
	private function gifAddHeader() {
		$cmap = 0;
		if (ord($this->frameSources[0][10]) & 0x80) {

			$cmap = 3 * (2 << (ord($this->frameSources[0][10]) & 0x07));

			$this->gif .= substr($this->frameSources[0], 6, 7);
			$this->gif .= substr($this->frameSources[0], 13, $cmap);
			$this->gif .= "!\377\13NETSCAPE2.0\3\1".$this->encodeAsciiToChar($this->loop)."\0";
		}
	}

	/**
     * Add the frame sources to the GIF string
     * 
     * @param integer $i
     * @param integer $d
     */
	private function addGifFrames($i, $d) {
		$Locals_str = 13 + 3 * (2 << (ord($this->frameSources[$i][10]) & 0x07));

		$Locals_end = strlen($this->frameSources[$i]) - $Locals_str - 1;
		$Locals_tmp = substr($this->frameSources[$i], $Locals_str, $Locals_end);

		$Global_len = 2 << (ord($this->frameSources[ 0][10]) & 0x07);
		$Locals_len = 2 << (ord($this->frameSources[$i][10]) & 0x07);

		$Global_rgb = substr($this->frameSources[ 0], 13, 3 * (2 << (ord($this->frameSources[ 0][10]) & 0x07)));
		$Locals_rgb = substr($this->frameSources[$i], 13, 3 * (2 << (ord($this->frameSources[$i][10]) & 0x07)));

		$Locals_ext = "!\xF9\x04".chr(($this->dis << 2) + 0).chr(($d >> 0 ) & 0xFF).chr(($d >> 8) & 0xFF)."\x0\x0";

		if ($this->colour > -1 && ord($this->frameSources[$i][10]) & 0x80) {

			for ($j = 0; $j < (2 << (ord($this->frameSources[$i][10]) & 0x07)); $j++) {

				if (ord($Locals_rgb[3 * $j + 0]) == (($this->colour >>16) & 0xFF) &&
					ord($Locals_rgb[3 * $j + 1]) == (($this->colour >> 8) & 0xFF) &&
					ord($Locals_rgb[3 * $j + 2]) == (($this->colour >> 0) & 0xFF)
				) {
					$Locals_ext = "!\xF9\x04".chr(($this->dis << 2) + 1).chr(($d >> 0) & 0xFF).chr(($d >> 8) & 0xFF).chr($j)."\x0";
					break;
				}
			}
		}

		switch ($Locals_tmp[0]) {
			case '!':
				$Locals_img = substr($Locals_tmp, 8, 10);
				$Locals_tmp = substr($Locals_tmp, 18, strlen($Locals_tmp) - 18);
			break;
			case ',':
				$Locals_img = substr($Locals_tmp, 0, 10);
				$Locals_tmp = substr($Locals_tmp, 10, strlen($Locals_tmp) - 10);         
			break;
		}

		if (ord($this->frameSources[$i][10]) & 0x80 && $this->imgBuilt) {

			if ($Global_len == $Locals_len) {

				if ($this->gifBlockCompare($Global_rgb, $Locals_rgb, $Global_len)) {
					$this->gif .= $Locals_ext.$Locals_img.$Locals_tmp;
				} else {
					$byte = ord($Locals_img[9]);
					$byte |= 0x80;
					$byte &= 0xF8;
					$byte |= (ord($this->frameSources[0][10]) & 0x07);
					$Locals_img[9] = chr($byte);
					$this->gif .= $Locals_ext.$Locals_img.$Locals_rgb.$Locals_tmp;
				}

			} else {
				$byte = ord($Locals_img[9]);
				$byte |= 0x80;
				$byte &= 0xF8;
				$byte |= (ord($this->frameSources[$i][10]) & 0x07);
				$Locals_img[9] = chr($byte);
				$this->gif .= $Locals_ext.$Locals_img.$Locals_rgb.$Locals_tmp;
			}

		} else {
			$this->gif .= $Locals_ext.$Locals_img.$Locals_tmp;
		}

		$this->imgBuilt = true;
	}

	/**
     * Add the gif string footer char
     */
	private function gifAddFooter() {
		$this->gif .= ';';
	}

	/**
     * Compare two blocks and return equality
     * 
     * @param string $globalBlock
     * @param string $localBlock
     * @param integer $length
     * 
     * @return bool Equality
	 */
	private function gifBlockCompare($globalBlock, $localBlock, $length) {
		for ($i = 0; $i < $length; $i++) {
			$p = 3 * $i;
			if ($globalBlock[$p + 0] != $localBlock[$p + 0] ||
				$globalBlock[$p + 1] != $localBlock[$p + 1] ||
				$globalBlock[$p + 2] != $localBlock[$p + 2])	{
                return false;
			}
		}
		return true;
	}

	/**
     * Encode an ASCII char into a string char (old: GIFWord)
     * 
     * $param integer $char ASCII char
     * 
     * @return string
	 */
	public function encodeAsciiToChar($char) {
		return (chr($char & 0xFF).chr(($char >> 8) & 0xFF));
	}

    /**
     * Reset and clean the current object
     */
    public function reset() {
        $this->frameSources;
        $this->gif = 'GIF89a'; // the GIF header
        $this->imgBuilt = false;
        $this->loop = 0;
        $this->dis = 2;
        $this->colour = -1;
		$this->ErrNo = 0;
    }

    // Getter / Setter
    // ===================================================================================
	/**
     * Get the final GIF image string (binary source)
     * 
     * @return string
	 */
	public function getGif() {
		return $this->gif;
	}

	/**
	 * Get the error message
	 *
	 * @return string: the message
	 */
	 public function error() {
		 return $this->errors[$this->ErrNo];
	 }
}